#!/usr/bin/env python
"""
The purpose of this file is to provide some high-functionality to the MergeBot
while hiding the details of the task tracking software.

This implementation uses Trac as the task tracker.
"""

import os
import time

import trac.env
import trac.ticket

class Task(object):
    "A convenience wrapper for Trac's ticket objects."
    def __init__(self, tracEnv, ticketnum):
        self.tracEnv = tracEnv
        self.tracTicket = trac.ticket.Ticket(self.tracEnv, ticketnum)
        self.comments = []
    def AddComment(self, comment):
        "Add a comment to the list of comments to add to the ticket."
        self.comments.append(comment)
    def SetMergeBotStatus(self, state):
        self.tracTicket['mergebotstate'] = state
    def CloseAsFixed(self):
        self.tracTicket['status'] = 'closed'
        self.tracTicket['resolution'] = 'fixed'
    def Save(self):
        comment = "\n\n".join(self.comments)
        try:
            self.tracTicket.save_changes("mergebot", comment)
        except: #IntegrityError:
            # Two changes too close together?  Try again, but just once, so we
            # don't get an infinite loop.
            time.sleep(1)
            self.tracTicket.save_changes("mergebot", comment)
    def GetSummary(self):
        return self.tracTicket['summary']

def GetWorkDir(tracEnv, ticket=None, module=None):
    """Generate a work directory name based on the ticket and/or module"""
    workdir = tracEnv.config.get("mergebot", "work_dir")
    if not os.path.isabs(workdir):
        workdir = os.path.join(os.path.abspath(tracEnv.path), workdir)
    nameparts = []
    if module:
        nameparts.append(module)
    if ticket:
        nameparts.append("ticket-%s" % (ticket, ))
    if nameparts:
        workdir = os.path.join(workdir, ".".join(nameparts))
    return workdir

def GetLogFile(tracEnv, ticket=None):
    if ticket:
        filename = "ticket-%s.log" % (ticket, )
    else:
        filename = "mergebot.log"
    return os.path.join(GetWorkDir(tracEnv), filename)

def GetRepositoryPublicUrl(tracEnv):
    return tracEnv.config.get("mergebot", "repository_url")

def GetRepositoryLocalUrl(tracEnv):
    dirname = tracEnv.config.get("trac", "repository_dir")
    if not os.path.isabs(dirname):
        dirname = os.path.join(os.path.abspath(tracEnv.path), dirname)
    return "file://%s" % (dirname, )
    
# vim:foldmethod=indent foldcolumn=8
# vim:softtabstop=4 shiftwidth=4 tabstop=4 expandtab
