#!/usr/bin/env python
"""
Syntax: MergeActor.py ticketnum component version requestor

Merge a branch to its trunk.
"""

import os
import sys
import trac.env

import SvnOps
from WorkQueue import MergeBotActor, VersionToDir
from TrackerTools import GetWorkDir, GetRepositoryLocalUrl, Task, GetLogFile

def merge_action(trac_env, ticketnum, component, version, requestor):
    "Merge a branch to its trunk"
    task_obj = Task(trac_env, ticketnum)
    logfile = GetLogFile(trac_env, ticketnum)

    checkoutdir = GetWorkDir(trac_env, ticketnum, __name__)
    # FIXME: Should we just bail out instead?
    if os.path.exists(checkoutdir):
        os.system("rm -rf \"%s\" >>%s 2>&1" % (checkoutdir, logfile))

    sourceurl = os.path.join(GetRepositoryLocalUrl(trac_env), component,
        VersionToDir(version))
    ticketurl = os.path.join(GetRepositoryLocalUrl(trac_env), component, "branches",
        "ticket-%s" % ticketnum)

    # FIXME: needs error checking
    startrev, endrev = SvnOps.get_branch_info(ticketurl, logfile)

    SvnOps.checkout(sourceurl, checkoutdir, logfile)
    # FIXME: check return code
    results = SvnOps.merge(ticketurl, checkoutdir, (startrev, endrev), logfile)
    conflicts = SvnOps.conflicts_from_merge_results(results)
    if conflicts:
        message = "\n".join([
            "Found %s conflicts in attempt to merge %s:%s to %s for %s." % \
                (len(conflicts), startrev, endrev, version, requestor),
            "Files in conflict:",
            "{{{",
            "\n".join(conflicts),
            "}}}",
            "A rebranch will be needed before this can be merged.",
        ])
        endstatus = "conflicts"
    else:
        # The merge worked, so commit the changes.
        summary = task_obj.GetSummary()
        commitmessage = "\n".join([
            "Ticket #%s: %s" % (ticketnum, summary),
            "    Merge of %s:%s to %s for %s." % (startrev, endrev, version,
                requestor),
        ])
        committedrev = SvnOps.commit(checkoutdir, commitmessage, logfile)
        # Sed message and endstatus
        if committedrev == None:
            # Apparently nothing to commit.
            message = "\n".join([
                "Merged %s:%s to %s for %s." % (startrev, endrev, version,
                    requestor),
                "No changes to commit.",
            ])
            endstatus = "merged"
        elif committedrev >= 0:
            # The commit succeeded.
            message = "\n".join([
                "Merged %s:%s to %s for %s." % (startrev, endrev, version,
                    requestor),
                "Changeset [%s]. [source:%s/%s@%s]" % (committedrev,
                    component, VersionToDir(version), committedrev),
            ])
            endstatus = "merged"
        else:
            # The commit for the merge failed.
            message = "Commit failed in attempt to merge %s:%s to %s for %s." \
                % (startrev, endrev, version, requestor)
            endstatus = "mergefailed"

    # Clean up the work area
    os.system("rm -rf \"%s\" >>%s 2>&1" % (checkoutdir, logfile))

    task_obj.AddComment(message)
    return endstatus, task_obj

class MergeActor(MergeBotActor):
    "Actor wrapper for merge_action"
    def __init__(self, trac_env):
        MergeBotActor.__init__(self, trac_env, "merge", merge_action)

def main():
    tracdir = sys.argv[1]
    trac_env = trac.env.open_environment(tracdir)
    mergingActor = MergeActor(trac_env)
    mergingActor.AddTask(sys.argv[2:])
    mergingActor.Run()

if __name__ == "__main__":
    main()

# vim:foldcolumn=4 foldmethod=indent
# vim:tabstop=4 shiftwidth=4 softtabstop=4 expandtab
