#!/usr/bin/env python
"""Module for creating new branches for tickets"""
# Syntax: BranchActor.py ticketnum component version requestor

import os
import sys
import time
import trac.env

import SvnOps
from WorkQueue import MergeBotActor, VersionToDir
from TrackerTools import GetRepositoryPublicUrl, GetRepositoryLocalUrl, Task, \
    GetLogFile

def branch_action(trac_env, ticketnum, component, version, requestor):
    "Create the branch"
    task_obj = Task(trac_env, ticketnum)

    # Setup logging
    logfile = GetLogFile(trac_env, ticketnum)
    open(logfile, "a").write("%s: branching ticket %s\n" % (time.asctime(),
        ticketnum))

    # Determine the URLS to copy from and to
    branchdir = "branches/ticket-%s" % (ticketnum)
    copyfrom = os.path.join(GetRepositoryLocalUrl(trac_env), component,
        VersionToDir(version))
    copyto = os.path.join(GetRepositoryLocalUrl(trac_env), component, branchdir)

    commit_header = "Ticket #%s: %s" % (ticketnum, task_obj.GetSummary())

    # Delete the branch if it already exists.  This can happen if the branch
    # was merged, but we're still working on it.
    if SvnOps.get_branch_info(copyto, logfile):
        # This branch already exists.
        commit_message = "\n".join([commit_header,
            "    Delete old branch",
        ])
        new_rev = SvnOps.delete_branch(copyto, commit_message, logfile)
        if new_rev == -1:
            status = "branchfailed"
            return status, task_obj

    # Do the branch creationg
    commit_message = "\n".join([commit_header,
        "    Create branch from %s for %s." % (version, requestor),
    ])
    retval = SvnOps.create_branch(copyfrom, copyto, commit_message, logfile)
    if retval:
        # Failed for some reason.
        status = "branchfailed"
    else:
        publiccopyto = os.path.join(GetRepositoryPublicUrl(trac_env), component,
            branchdir)
        comment = "\n".join([
            "Created branch from %s for %s." % (version, requestor),
            "",
            "Browse branch [source:%s source code] and [log:%s commit log]." % 
                (os.path.join(component, branchdir),
                os.path.join(component, branchdir)),
            "",
            "To checkout, run:",
            "{{{",
            "svn checkout %s %s-%s" % (publiccopyto, component, ticketnum),
            "}}}",
        ])
        task_obj.AddComment(comment)
        status = "branched"
    return status, task_obj

class BranchActor(MergeBotActor):
    "Actor for creating a new branch."
    def __init__(self, trac_env):
        MergeBotActor.__init__(self, trac_env, "branch", branch_action)

def main():
    tracdir = sys.argv[1]
    trac_env = trac.env.open_environment(tracdir)
    branchingActor = BranchActor(trac_env)
    branchingActor.AddTask(sys.argv[2:])
    branchingActor.Run()

if __name__ == "__main__":
    main()

# vim:foldcolumn=4 foldmethod=indent
# vim:tabstop=4 shiftwidth=4 softtabstop=4 expandtab
