#!/usr/bin/env python
"""
Verify that a branch can be merged to its trunk without conflicts, but don't
commit the merge.
"""

import os
import shutil

from mergebot import SvnOps
from mergebot.Actor import Actor

class CheckMergeActor(Actor):
    """Checks that this ticket can be merged to its baseline, but don't modify
    the repository.
    """
    def execute(self):
        """
        Verify that a branch can be merged to its trunk without conflicts, but
        don't commit the merge.
        """
        results = {}
        workdir = self.work_dir
        logfile = self.logfilename()

        if os.path.exists(workdir):
            shutil.rmtree(workdir)

        # Make sure the various urls we require do exist
        if not SvnOps.get_branch_info(self.local_url(), logfile):
            comment = 'Component %s does not exist in the repository.' \
                % self.component
            return results, comment, False
        if not SvnOps.get_branch_info(self.local_url() + '/branches', logfile):
            comment = 'No directory in which to create branches for ' \
                'component %s in the repository.' % self.component
            return results, comment, False
        if not SvnOps.get_branch_info(self.baseline_local_url(), logfile):
            comment = 'Version %s for component %s does not exist in the ' \
                'repository.' % (self.version, self.component)
            return results, comment, False

        branch_info = SvnOps.get_branch_info(self.branch_local_url(), logfile)
        if not branch_info:
            comment = 'Branch for ticket %s does not exist in the repository.' \
                % (self.ticket)
            return results, comment, False
        startrev, endrev = branch_info

        SvnOps.checkout(self.baseline_local_url(), workdir, logfile)
        # TODO: check return code of the above
        merge_results = SvnOps.merge(self.branch_local_url(), workdir,
                                     (startrev, endrev), logfile)
        conflicts = SvnOps.conflicts_from_merge_results(merge_results)
        if conflicts:
            message = '\n'.join([
                'Found %s conflicts while checking merge of %s:%s to %s for ' \
                    '%s.' % (len(conflicts), startrev, endrev, self.version,
                    self.requestor),
                'Files in conflict:',
                '{{{',
                '\n'.join(conflicts),
                '}}}',
                'A rebranch will be needed before this can be merged.',
            ])
            success = False
        else:
            message = 'Found no conflicts while checking merge of %s:%s to ' \
                '%s for %s.' % (startrev, endrev, self.version, self.requestor)
            success = True

        # Clean up the work area
        if os.path.exists(workdir):
            shutil.rmtree(workdir)

        return results, message, success

# vim:foldcolumn=4 foldmethod=indent
# vim:tabstop=4 shiftwidth=4 softtabstop=4 expandtab
